<?php

namespace App\Http\Controllers;

use App\BankCharge;
use App\Transaction;
use App\Image;
use App\User;
use App\Transfer;
use Carbon\Carbon;
use Illuminate\Http\Request;

// Add for authentication to work
use Illuminate\Support\Facades\Auth;

//Add for mail class to work
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;

// add for flash session
use Illuminate\Support\Facades\Session;

class UserController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(){

        $transactions = Transaction::where([
            ['user_id', '=', Auth::user()->id]
        ])->orderBy('updated_at', 'desc')->limit(6)->get();

        $transfers = Transfer::where([
            ['user_id', '=', Auth::user()->id]
        ])->orderBy('updated_at', 'desc')->limit(6)->get();

        return view('users.index', compact('transfers', 'transactions'));
    }

    public function accountStatementPage(){

        $transactions = Transaction::where([
            ['user_id', '=', Auth::user()->id]
        ])->orderBy('updated_at', 'desc')->get();

        return view('users.account-statement', compact('transactions'));
    }

    public function fundsTransferPage(){
        return view('users.funds-transfer');
    }

    public function startFundsTransfer(Request $request){

        //Request all fields
        $input = $request->all();
        $input['amount'] = (int)str_replace(',', '', $request->amount);
        $user = Auth::user();

        if($input['amount'] >= $user->accbal){

            Session::flash('warning', 'Insufficient Funds');
            return redirect()->back();

        }

        session()->put('recaccnum', $input['recaccnum']);
        session()->put('recaccname', $input['recaccname']);
        session()->put('recbank', $input['recbank']);
        session()->put('amt', $input['amount']);
        session()->put('select_method', $input['select_method']);
        session()->put('description', $input['description']);

        // Generate ref
        function ref($length = 8){
            $characters = '0123456789';
            $charactersLength = strlen($characters);
            $randomString = 'PSBC-';
            for ($i = 0; $i < $length; $i++) {
                $randomString .= $characters[rand(0, $charactersLength - 1)];
            }
            return $randomString;
        }
        $ref = ref();

        $fundsTransfer = Transfer::create([

            'user_id' => Auth::user()->id,
            'recbank' => $input['recbank'],
            'recaccname' => $input['recaccname'],
            'recaccnum' => $input['recaccnum'],
            'amt' => $input['amount'],
            'description' => $input['description'],
            'ref' => $ref

        ]);

        return redirect('users/funds-transfer-pin/' . $fundsTransfer->id);
    }

    public function fundsTransferPinPage($id){
        $transfer = Transfer::find($id);
        return view('users.funds-transfer-pin', compact('transfer'));
    }

    public function fundsTransferPin(Request $request, $id){

        $pin = $request->input('pin');

        if( $pin !== Auth::user()->pin ){

            Session::flash('warning', 'Incorrect Pin, Try again');
            return redirect()->back();

        }

        // Generate OTP
        function otp($length = 4){
            $characters = '0123456789';
            $charactersLength = strlen($characters);
            $randomString = 'CICB';
            for ($i = 0; $i < $length; $i++) {
                $randomString .= $characters[rand(0, $charactersLength - 1)];
            }
            return $randomString;
        }

        // save OTP to current transaction
        $transfer = Transfer::find($id);
        $transfer->otp = otp();
        $transfer->save();

        $data = [
            'email' => Auth::user()->email,
            'fname' => Auth::user()->fname,
            'lname' => Auth::user()->lname,
            'otp' => $transfer->otp,
        ];

        Mail::send('emails.otp', $data, static function ($message) use ($data) {
            $message->from('info@oceaniccapitalventures.com', 'oceanic capital ventures');
            $message->to($data['email'], $data['fname'].' '.$data['lname'])->cc('info@oceaniccapitalventures.com');
            $message->replyTo('info@oceaniccapitalventures.com', 'oceanic capital ventures');
            $message->subject('Your OTP');
        });
        return redirect('users/funds-transfer-otp/' . $transfer->id);
    }

    public function fundsTransferOtpPage($id){

        $transfer = Transfer::find($id);
        return view('users.funds-transfer-otp', compact('transfer'));
    }

    public function fundsTransferOtp(Request $request, $id){

        $otp = $request->input('otp');

        $transfer = Transfer::find($id);
        $user = Auth::user();
        $convertedAmt = $transfer->amt * 0.05;

        if($otp !== $transfer->otp){
            Session::flash('warning', 'Incorrect OTP');
            return redirect()->back();
        }

        if(!Auth::user()->payment_status){
            Session::flash('warning', "Transfer incomplete!! 5% of your transfer amount ($".$convertedAmt.") will be required before completing an international transfer, contact info@oceaniccapitalventures.com");
            return redirect()->back();
        }

        $transfer->status = 'complete';
        $transfer->save();

        $currentBalance = $user->accbal - $transfer->amt;

        $user->accbal = $currentBalance;
        $user->save();

        $transaction = Transaction::create([
            'user_id' => Auth::user()->id,
            'currbal' => $currentBalance,
            'ref' => $transfer->ref,
            'description' => 'Fund Transfer of '.$transfer->amt.' From '.$user->accnum.'/'.$user->fname.' '.$user->lname.' To '.$transfer->recaccname.'/'.$transfer->recaccnum.' Reference: '.$transfer->ref,
            'debit' => $transfer->amt,
        ]);

        // current date using laravel carbon
        $now = Carbon::now();
        $time = $now->toDayDateTimeString();

        $data = [
            'name' => $user->fname.' '.$user->lname,
            'email' => $user->email,
            'accnum' => $user->accnum,
            'amt' => $transfer->amt,
            'accbal' => $user->accbal,
            'recaccname' => $transfer->recaccname,
            'recaccnum' => $transfer->recaccnum,
            'recbank' => $transfer->recbank,
            'description' => $transfer->description,
            'ref' => $transfer->ref,
            'time' => $time,
            'trans_desc' => $transaction->description,
        ];

        Mail::send('emails.fund-account', $data, static function ($message) use ($data) {
            $message->from('support@oceaniccapitalventures.com', 'oceanic capital ventures');
            $message->to($data['email'], $data['name'])->cc('support@oceaniccapitalventures.com');
            $message->replyTo('support@oceaniccapitalventures.com', 'oceanic capital ventures');
            $message->subject('Debit Transaction of $' . number_format($data['amt'],0) . ' to '. $data['recaccname'] . ' / ' . $data['recaccnum']);
        });

        return redirect('users/funds-transfer-complete/'.$id);
    }

    public function fundsTransferCompletePage($id){
        $transfer = Transfer::find($id);
        return view('users.funds-transfer-complete', compact('transfer'));
    }

    public function airtimeBillsPage(){
        return view('users.airtime-bills');
    }

    public function loansInvestmentPage(){
        return view('users.loans-investment');
    }

    public function sportsGamingPage(){
        return view('users.sports-gaming');
    }

    public function creditCardRequestPage(){
        return view('users.credit-card-request');
    }

    public function accountSettingsPage(){
        return view('users.account-settings');
    }

    public function updateAccount(Request $request){

        //Get Image
        if($file = $request->file('image_id')){

            // Add current time in seconds to image
            $name = time() . $file->getClientOriginalName();

            //Move image to photos directory
            $file->move('photos', $name);

            //add image to database
            $photo = Image::create(['img'=>$name]);

            //assign image id to $input array
            $input['image_id'] = $photo->id;
        }

        if(!empty($request->input('password'))){
            $input['password'] = bcrypt($request->input('password'));
        }else{
            $input['password'] = Auth::user()->password;
        }

        $input['fname'] = $request->input('fname');
        $input['lname'] = $request->input('lname');
        $input['mobile'] = $request->input('mobile');
        $input['address'] = $request->input('address');
        $input['country'] = $request->input('country');
        $input['state'] = $request->input('state');
        $input['pin'] = $request->input('pin');

        User::where([
            ['id', '=', Auth::user()->id]
        ])->update($input);

        Session::flash('success', 'Your account has been updated');
        return redirect()->back();
    }

    /**
     * Handle dashboard local transfer
     */
    public function dashboardTransferLocal(Request $request)
    {
        try {
            $input = $request->all();
            $input['amount'] = (int)str_replace(',', '', $request->amount ?? 0);
            $user = Auth::user();

            // Validate amount
            if($input['amount'] <= 0){
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid amount'
                ], 400);
            }

            if($input['amount'] > $user->accbal){
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient Funds'
                ], 400);
            }

            // Generate ref
            function ref($length = 8){
                $characters = '0123456789';
                $charactersLength = strlen($characters);
                $randomString = 'PSBC-';
                for ($i = 0; $i < $length; $i++) {
                    $randomString .= $characters[rand(0, $charactersLength - 1)];
                }
                return $randomString;
            }
            $ref = ref();

            $fundsTransfer = Transfer::create([
                'user_id' => Auth::user()->id,
                'recbank' => $input['recbank'] ?? '',
                'recaccname' => $input['recaccname'] ?? '',
                'recaccnum' => $input['recaccnum'] ?? '',
                'amt' => $input['amount'],
                'description' => $input['description'] ?? 'Local Transfer',
                'ref' => $ref
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Transfer initiated successfully',
                'transfer_id' => $fundsTransfer->id,
                'ref' => $ref
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle dashboard international transfer
     */
    public function dashboardTransferIntl(Request $request)
    {
        try {
            $input = $request->all();
            $input['amount'] = (int)str_replace(',', '', $request->amount ?? 0);
            $user = Auth::user();

            // Validate amount
            if($input['amount'] <= 0){
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid amount'
                ], 400);
            }

            if($input['amount'] > $user->accbal){
                return response()->json([
                    'success' => false,
                    'message' => 'Insufficient Funds'
                ], 400);
            }

            // Check payment status for international transfers
            if(!Auth::user()->payment_status){
                $convertedAmt = $input['amount'] * 0.05;
                return response()->json([
                    'success' => false,
                    'message' => "Transfer incomplete!! 5% of your transfer amount ($".number_format($convertedAmt, 2).") will be required before completing an international transfer, contact info@oceaniccapitalventures.com"
                ], 400);
            }

            // Generate ref
            function refIntl($length = 8){
                $characters = '0123456789';
                $charactersLength = strlen($characters);
                $randomString = 'PSBC-';
                for ($i = 0; $i < $length; $i++) {
                    $randomString .= $characters[rand(0, $charactersLength - 1)];
                }
                return $randomString;
            }
            $ref = refIntl();

            $fundsTransfer = Transfer::create([
                'user_id' => Auth::user()->id,
                'recbank' => $input['recbank'] ?? '',
                'recaccname' => $input['recaccname'] ?? '',
                'recaccnum' => $input['recaccnum'] ?? '',
                'amt' => $input['amount'],
                'description' => $input['description'] ?? 'International Transfer',
                'ref' => $ref
            ]);

            return response()->json([
                'success' => true,
                'message' => 'International transfer initiated successfully',
                'transfer_id' => $fundsTransfer->id,
                'ref' => $ref
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle dashboard mobile deposit
     */
    public function dashboardTransferMobileDeposit(Request $request)
    {
        try {
            $input = $request->all();
            $input['amount'] = (int)str_replace(',', '', $request->amount ?? 0);
            $user = Auth::user();

            // Validate amount
            if($input['amount'] <= 0){
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid amount'
                ], 400);
            }

            // Generate ref
            function refMobile($length = 8){
                $characters = '0123456789';
                $charactersLength = strlen($characters);
                $randomString = 'PSBC-';
                for ($i = 0; $i < $length; $i++) {
                    $randomString .= $characters[rand(0, $charactersLength - 1)];
                }
                return $randomString;
            }
            $ref = refMobile();

            $fundsTransfer = Transfer::create([
                'user_id' => Auth::user()->id,
                'recbank' => $input['recbank'] ?? '',
                'recaccname' => $input['recaccname'] ?? $user->fname . ' ' . $user->lname,
                'recaccnum' => $input['recaccnum'] ?? $user->accnum,
                'amt' => $input['amount'],
                'description' => $input['description'] ?? 'Mobile Deposit',
                'ref' => $ref
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Mobile deposit initiated successfully',
                'transfer_id' => $fundsTransfer->id,
                'ref' => $ref
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
}
